import { ref } from 'vue';
import useMonthsList from '@/functions/MonthsList';
import Highcharts from 'highcharts';
import type {ModalType} from "~/types/core/ModalType";

export const useChart = (product:Record<string,string>) => {

    const theme = useState('theme');
    const chartBox = ref<ModalType|null>(null);
    const months:Record<number,string> = useMonthsList();
    const global = useNuxtApp().vueApp.config.globalProperties;
    const dataByDate:Record<string, any>={};
    const selectedVariation = ref<number | null>(null);
    const {api}=useRuntimeConfig().public;
    const {$axios}=useNuxtApp();
    const loading=ref(false);
    const toast=useState('toast');

    interface ChartDataType {
        [p: string]: {
            variation: Record<string, string>;
            changes:{
                price:number,
                time:number
            }[];
        };
    }

    const data = ref<ChartDataType>({});

    const chartOptions = ref({
        chart: {
            type: 'line',
            style: {
                fontFamily: 'inherit',
            },
            height:theme.value=='desktop' ? 350 : 'auto'
        },
        title: {
            text: '',
        },
        yAxis: {
            title: { text: '' },
            labels: {
                formatter(this: Highcharts.AxisLabelsFormatterContextObject) {
                    let value = this.value as string;
                    value = global.$replaceEnNumber(global.$numberFormat(value));
                    return `<div style="direction:rtl"><span>${value}</span><span> تومان </span></div>`;
                },
                style: {
                    fontSize: '12px',
                },
                useHTML: true,
            },
            tickPositioner: function (this: Highcharts.Axis): (number | string)[] {
                const max = this.max as number;
                const min = this.min as number;
                const range = max - min;

                const step = Math.ceil(range / 4);

                const ticks: (number | string)[] = [];
                for (let i = min; i <= max; i += step) {
                    ticks.push(i);
                }
                return ticks;
            }
        },
        xAxis: {
            categories: []  as string[],
            crosshair: {
                color: 'rgba(236,235,235,0.54)',
                width:20,
                zIndex: 3
            },
            labels: {
                formatter: function (this: any) {
                    const categories = chartOptions.value.xAxis.categories;
                    if (this.isFirst || this.isLast) {
                        const category = global.$replaceEnNumber(categories[this.pos]);
                        return `<span style="direction:rtl;">${category}</span>`;
                    }
                    return '';
                },
                useHTML: true,
            },
        },
        tooltip: {
            useHTML: true,
            padding:0,
            shadow: false,
            formatter(this: Highcharts.TooltipFormatterContextObject) {
                let html='<div class="price-chart-tooltip">';
                if(dataByDate[this.x as string] && dataByDate[this.x as string].seller){
                    html+=`<p>
                           <img alt="" width="15px" src="${useRuntimeConfig().public.appUrl+'/images/house-solid.svg'}"/>
                           <span style="padding:0 5px">${dataByDate[this.x as string].seller.brandName}</span>
                           </p>`;
                }
                if(dataByDate[this.x as string] && dataByDate[this.x as string].param2){
                    html+=`<p>
                          <img alt="" width="15px" src="${useRuntimeConfig().public.appUrl+'/images/circle-check-solid.svg'}"/>
                          <span style="padding:0 5px">${dataByDate[this.x as string].param2.name}</span>
                    </p>`;
                }
                html+=`<p style="font-weight:bold;font-size:15px">${global.$replaceEnNumber(global.$numberFormat(this.y))} تومان</p>` +
                  `<p style="font-size:13px;color:gray">${global.$replaceEnNumber(global.$getDate(dataByDate[this.x as string].time))} </p>` +
                '</div>';
                return html;
            },
            style: { fontSize: '14px'}
        },
        plotOptions: {
            series: {
                marker: {
                    enabled: false,
                },
            },
        },
        series: [
            {
                name: '',
                data:[] as number[],
            },
        ],
        legend: {
            enabled: false
        }
    });

    const getChartData = () => {
        loading.value=true;
        $axios.get(api + '/product/' + product.id + '/changes-chart')
            .then((response) => {
                data.value = response.data;
                if(data.value &&  Object.keys(data.value).length>0){
                    chartBox.value?.show();
                    const keys = Object.keys(data.value);
                    if(!selectedVariation.value && data.value[keys[0]]){
                        selectedVariation.value = parseInt(keys[0]);
                        updateChart();
                    }
                }
                else{
                    toast.value={message:'تغییرات قیمت طی یک ماه برای این کالا ثبت نشده'};
                }
            })
            .finally(()=>{
            loading.value=false;
        });
    };

    const updateChart = () => {
        const changes = data.value[selectedVariation.value!]?.changes || [];
        const times: string[] = [];
        const chartValues: number[] = [];
        const zones: any[] = [];

        let lastValue: number = 0;

        changes.forEach((row, index) => {
            const dateData = global.$getDate(row.time);
            const [_, month, day] = dateData.toString().split('/');
            times.push( months[month]+'-'+day);
            dataByDate[months[month]+'-'+day]=row;
            const price = row.price;

            if (price === 0 && lastValue !== null) {
                chartValues.push(lastValue);
                zones.push({
                    value: index,
                    color: '#bbbbbb',
                });
            } else {
                chartValues.push(price);
                lastValue = price;
                zones.push({
                    value: index,
                    color: '#4BBFD3',
                });
            }
        });

        chartOptions.value.xAxis.categories = times;
        chartOptions.value.series = [
            {
                name: 'قیمت',
                data: chartValues,
                zoneAxis: 'x',
                zones: zones,
            },
        ];
    };

    const handleVariationClick = (key: number) => {
        selectedVariation.value = key;
        updateChart();
    };

    return {
        theme,
        chartBox,
        chartOptions,
        data,
        selectedVariation,
        loading,
        getChartData,
        handleVariationClick,
    };
};
