export function useAudioRecorder() {
    const isRecording = ref(false)
    const duration = ref(0)
    const audioUrl = ref<string | null>(null)
    const audioBlob = ref<Blob | null>(null)

    const audioChunks: Blob[] = []
    let mediaRecorder: MediaRecorder | null = null
    let timerInterval: ReturnType<typeof setInterval> | null = null

    const formatTime = (seconds: number): string => {
        const m = Math.floor(seconds / 60).toString().padStart(2, '0')
        const s = (seconds % 60).toString().padStart(2, '0')
        return `${m}:${s}`
    }

    const startRecording = async (onError?: (msg: string) => void) => {
        try {
            const stream = await navigator.mediaDevices.getUserMedia({ audio: true })
            mediaRecorder = new MediaRecorder(stream)
            audioChunks.length = 0

            mediaRecorder.ondataavailable = (e: BlobEvent) => {
                audioChunks.push(e.data)
            }

            mediaRecorder.onstop = () => {
                const blob = new Blob(audioChunks, { type: 'audio/webm' })
                audioBlob.value = blob;
                audioUrl.value = URL.createObjectURL(blob)
            }

            mediaRecorder.start()
            isRecording.value = true
            duration.value = 0

            timerInterval = setInterval(() => {
                duration.value += 1
            }, 1000)
        } catch (err) {
            onError?.('میکروفن شما فعال نیست')
        }
    }

    const stopRecording = () => {
        mediaRecorder?.stop()
        isRecording.value = false
        if (timerInterval) clearInterval(timerInterval)
    }

    const toggleRecording = (onError?: (msg: string) => void) => {
        if (isRecording.value) {
            stopRecording()
        } else {
            startRecording(onError)
        }
    }

    onUnmounted(() => {
        if (timerInterval) clearInterval(timerInterval)
    })

    return {
        isRecording,
        duration,
        audioUrl,
        audioBlob,
        formatTime,
        toggleRecording,
        stopRecording
    }
}
